<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>File Download Manager</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            color: #333;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f5f5f5;
        }
        .container {
            background-color: #fff;
            border-radius: 8px;
            padding: 25px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }
        h1 {
            color: #2c3e50;
            text-align: center;
            margin-bottom: 30px;
        }
        .code-block {
            background-color: #f8f9fa;
            border-left: 4px solid #3498db;
            padding: 15px;
            overflow-x: auto;
            margin: 20px 0;
            font-family: 'Courier New', monospace;
            font-size: 14px;
        }
        .issue {
            background-color: #fff4f4;
            border-left: 4px solid #e74c3c;
            padding: 15px;
            margin: 20px 0;
        }
        .solution {
            background-color: #f0fff4;
            border-left: 4px solid #2ecc71;
            padding: 15px;
            margin: 20px 0;
        }
        .note {
            background-color: #f0f8ff;
            border-left: 4px solid #3498db;
            padding: 15px;
            margin: 20px 0;
        }
        .button {
            display: inline-block;
            background-color: #3498db;
            color: white;
            padding: 10px 20px;
            text-decoration: none;
            border-radius: 4px;
            font-weight: bold;
            margin-top: 10px;
        }
        .button:hover {
            background-color: #2980b9;
        }
        .file-list {
            list-style-type: none;
            padding: 0;
        }
        .file-list li {
            padding: 10px;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .file-list li:last-child {
            border-bottom: none;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>File Download Manager</h1>
        
        <div class="issue">
            <h2>Issue Detected</h2>
            <p>The PHP code has several problems:</p>
            <ol>
                <li>Using a URL (<code>http://127.0.0.1/...</code>) as a file path for <code>file_exists()</code> and <code>filesize()</code></li>
                <li>Syntax error with the <code>die</code> statement (missing parenthesis)</li>
                <li>Potential security issues with direct file path usage</li>
            </ol>
        </div>
        
        <div class="solution">
            <h2>Corrected PHP Code</h2>
            <p>Here's the fixed version of your PHP code:</p>
            <div class="code-block">
                &lt;?php<br>
                include("../jc/app-config/DB_config.php");<br>
                <br>
                if (isset($_GET['id'])) {<br>
                &nbsp;&nbsp;&nbsp;&nbsp;$id = intval($_GET['id']);<br>
                <br>
                &nbsp;&nbsp;&nbsp;&nbsp;$sql = "SELECT file_path FROM bulletin_uploads WHERE id = ?";<br>
                &nbsp;&nbsp;&nbsp;&nbsp;$stmt = $conn->prepare($sql);<br>
                &nbsp;&nbsp;&nbsp;&nbsp;$stmt->bind_param("i", $id);<br>
                &nbsp;&nbsp;&nbsp;&nbsp;$stmt->execute();<br>
                &nbsp;&nbsp;&nbsp;&nbsp;$stmt->bind_result($filePath);<br>
                &nbsp;&nbsp;&nbsp;&nbsp;$stmt->fetch();<br>
                &nbsp;&nbsp;&nbsp;&nbsp;$stmt->close();<br>
                <br>
                &nbsp;&nbsp;&nbsp;&nbsp;// Extract just the filename from the path<br>
                &nbsp;&nbsp;&nbsp;&nbsp;$fileName = basename($filePath);<br>
                &nbsp;&nbsp;&nbsp;&nbsp;// Create the full server path to the file<br>
                &nbsp;&nbsp;&nbsp;&nbsp;$serverFilePath = "../jc/bulletin/" . $fileName;<br>
                <br>
                &nbsp;&nbsp;&nbsp;&nbsp;if ($serverFilePath && file_exists($serverFilePath)) {<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;// Get file info<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;$fileSize = filesize($serverFilePath);<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;// Get MIME type<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;$fileExtension = strtolower(pathinfo($serverFilePath, PATHINFO_EXTENSION));<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;$contentType = getContentType($fileExtension);<br>
                <br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;// Force download headers<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;header("Content-Type: $contentType");<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;header("Content-Disposition: attachment; filename=\"$fileName\"");<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;header("Content-Length: $fileSize");<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;header("Pragma: no-cache");<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;header("Expires: 0");<br>
                <br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;// Clear output buffer<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;ob_clean();<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;flush();<br>
                <br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;// Send file<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;readfile($serverFilePath);<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;exit;<br>
                &nbsp;&nbsp;&nbsp;&nbsp;} else {<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;die("File not found.");<br>
                &nbsp;&nbsp;&nbsp;&nbsp;}<br>
                } else {<br>
                &nbsp;&nbsp;&nbsp;&nbsp;die("Invalid request.");<br>
                }<br>
                <br>
                function getContentType($extension) {<br>
                &nbsp;&nbsp;&nbsp;&nbsp;$types = [<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'pdf' => 'application/pdf',<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'doc' => 'application/msword',<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'jpg' => 'image/jpeg',<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'jpeg' => 'image/jpeg',<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'png' => 'image/png',<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'gif' => 'image/gif',<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'txt' => 'text/plain',<br>
                &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;// Add more as needed<br>
                &nbsp;&nbsp;&nbsp;&nbsp;];<br>
                &nbsp;&nbsp;&nbsp;&nbsp;return isset($types[$extension]) ? $types[$extension] : 'application/octet-stream';<br>
                }<br>
                ?&gt;
            </div>
        </div>
        
        <div class="note">
            <h2>Key Changes Made</h2>
            <ul>
                <li>Fixed the file path issue by using the correct server path instead of a URL</li>
                <li>Corrected the syntax error with the <code>die("File not found.");</code> statement</li>
                <li>Added a function to determine proper Content-Type headers based on file extension</li>
                <li>Improved security by using proper file path handling</li>
            </ul>
        </div>
        
        <h2>Test File Downloads</h2>
        <p>Below is a simulation of how your file download system would work:</p>
        
        <ul class="file-list">
            <li>
                <span>Weekly Bulletin.pdf</span>
                <a href="#" class="button" onclick="simulateDownload(1)">Download</a>
            </li>
            <li>
                <span>Event Schedule.docx</span>
                <a href="#" class="button" onclick="simulateDownload(2)">Download</a>
            </li>
            <li>
                <span>Church Announcements.txt</span>
                <a href="#" class="button" onclick="simulateDownload(3)">Download</a>
            </li>
        </ul>
        
        <div id="download-status" style="margin-top: 20px; padding: 10px; display: none;"></div>
    </div>

    <script>
        function simulateDownload(fileId) {
            const statusDiv = document.getElementById('download-status');
            statusDiv.style.display = 'block';
            
            // Simulate different outcomes based on fileId
            if (fileId === 1) {
                statusDiv.innerHTML = '<div style="color: #2ecc71;">Download started successfully! (Simulation)</div>';
            } else if (fileId === 2) {
                statusDiv.innerHTML = '<div style="color: #2ecc71;">Download started successfully! (Simulation)</div>';
            } else {
                statusDiv.innerHTML = '<div style="color: #e74c3c;">Error: File not found. (Simulation)</div>';
            }
            
            // Hide status after 3 seconds
            setTimeout(() => {
                statusDiv.style.display = 'none';
            }, 3000);
        }
    </script>
</body>
</html>