<?php
// Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Include database configuration and Name class
include("app-config/DB_config.php");
$myName = new Name();

// Get the email from session if it exists
$emailing = isset($_SESSION['email']) ? $_SESSION['email'] : '';

// Check if 'value' parameter is set in the GET request
if (isset($_GET['value'])) {
    $value = $_GET['value'];

    /**
     * Function to fetch room details based on the provided room ID
     */
    function getRoomDetails($conn, $value) {
        $query = "SELECT `id`, `location_id`, `name`, `cost`, `description`, `adults`, `children`, `created_at`, `status`, `registered_by` 
                  FROM `rooms` 
                  WHERE `id` = ?";
        
        if ($stmt = $conn->prepare($query)) {
            $stmt->bind_param("i", $value);
            $stmt->execute();
            $result = $stmt->get_result();
            $stmt->close();
            return $result;
        } else {
            die("Error preparing statement: " . $conn->error);
        }
    }

    /**
     * Function to fetch room pictures based on the provided room ID
     */
    function getRoomPictures($conn, $room_id) {
        $query = "SELECT `id`, `room_id`, `picture_path`, `created_at` 
                  FROM `room_pictures` 
                  WHERE `room_id` = ?";
        
        if ($stmt = $conn->prepare($query)) {
            $stmt->bind_param("i", $room_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $stmt->close();
            return $result;
        } else {
            die("Error preparing statement: " . $conn->error);
        }
    }

    /**
     * Function to fetch room amenities based on the provided room ID
     */
    function getRoomAmenities($conn, $room_id) {
        $query = "SELECT a.`id`, a.`room_id`, a.`amenity_id`, a.`created_at`, am.`amenities` AS amenity_name
                  FROM `amenities` a
                  JOIN `room_amenities` am ON a.`amenity_id` = am.`id`
                  WHERE a.`room_id` = ?";
        
        if ($stmt = $conn->prepare($query)) {
            $stmt->bind_param("i", $room_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $stmt->close();
            return $result;
        } else {
            die("Error preparing statement: " . $conn->error);
        }
    }

    /**
     * Function to fetch room videos based on the provided room ID
     */
    function getRoomVideos($conn, $room_id) {
        $query = "SELECT `id`, `room_id`, `video_path`, `created_at` 
                  FROM `room_videos` 
                  WHERE `room_id` = ?";
        
        if ($stmt = $conn->prepare($query)) {
            $stmt->bind_param("i", $room_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $stmt->close();
            return $result;
        } else {
            die("Error preparing statement: " . $conn->error);
        }
    }

    // Fetch room details
    $roomDetailsResult = getRoomDetails($conn, $value);

    if ($roomDetailsResult->num_rows > 0) {
        $roomDetails = $roomDetailsResult->fetch_assoc();

        // Sanitize fetched data
        $id = htmlspecialchars_decode($roomDetails['id']);
        $location_id = htmlspecialchars_decode($roomDetails['location_id']);
        $name = htmlspecialchars_decode($roomDetails['name']);
        $cost = htmlspecialchars_decode($roomDetails['cost']);
        $description = htmlspecialchars_decode($roomDetails['description']);
        $adults = htmlspecialchars_decode($roomDetails['adults']);
        $children = htmlspecialchars_decode($roomDetails['children']);
        $created_at = htmlspecialchars_decode($roomDetails['created_at']);
        $status = htmlspecialchars_decode($roomDetails['status']);
        $registered_by = $roomDetails['registered_by'];

        // Set CSS class and status label based on the status value
        $statusCSS = $status == 1 ? "badge bg-grd-success" : "badge bg-grd-danger";
        $statusParam = $status == 1 ? "Active" : "Not Active";

        // Fetch the location name based on the location_id
        $location_name = $myName->showName($conn, "SELECT `name` FROM `location` WHERE `id` = '$location_id'");

        // Fetch room pictures
        $roomPicturesResult = getRoomPictures($conn, $id);
        $roomPictures = [];
        if ($roomPicturesResult->num_rows > 0) {
            while ($picture = $roomPicturesResult->fetch_assoc()) {
                $roomPictures[] = htmlspecialchars_decode($picture['picture_path']);
            }
        }

        // Fetch room amenities
        $roomAmenitiesResult = getRoomAmenities($conn, $id);
        $roomAmenities = [];
        if ($roomAmenitiesResult->num_rows > 0) {
            while ($amenity = $roomAmenitiesResult->fetch_assoc()) {
                $roomAmenities[] = htmlspecialchars_decode($amenity['amenity_name']);
            }
        }

        // Fetch room videos
        $roomVideosResult = getRoomVideos($conn, $id);
        $roomVideos = [];
        if ($roomVideosResult->num_rows > 0) {
            while ($video = $roomVideosResult->fetch_assoc()) {
                $roomVideos[] = htmlspecialchars_decode($video['video_path']);
            }
        }

        /**
         * Display room details, pictures, amenities, and videos in tables.
         */
//htmlspecialchars_decode_decode($roomDetails['description']);
        // Display room details
        echo "<h1>Room Details</h1>";
        echo "<table border='1' cellpadding='10' cellspacing='0' class='table table-striped table-bordered'>";
        echo "<tr><th width = '10%'>Room Name</th><td >{$name}</td></tr>";
        echo "<tr><th>Location</th><td>{$location_name}</td></tr>";
        echo "<tr><th>Cost</th><td>".number_format($cost, 2)."</td></tr>";
        //echo "<tr><th>Description</th><td colspan='2'>{$description}</td></tr>";
        echo "<tr><th>Adults</th><td>{$adults}</td></tr>";
        echo "<tr><th>Children</th><td>{$children}</td></tr>";
        echo "<tr><th>Status</th><td><span class='{$statusCSS}'>{$statusParam}</span></td></tr>";
        echo "<tr><th>Registered By</th><td>{$registered_by}</td></tr>";
        echo "<tr><th>Created At</th><td>{$created_at}</td></tr>";
        echo "<tr><th colspan = '2'>Description</th></tr>";
        echo "<tr><th colspan = '2'>{$description}</th></tr>";
        echo "</table>";

        // Display room pictures
       // Display room pictures with Bootstrap Carousel
echo "<h2>Room Pictures</h2>";
if (!empty($roomPictures)) {
    echo '
    <div class="col">
        <h6 class="mb-0 text-uppercase">Room Pictures</h6>
        <hr>
        <div class="card">
            <div class="card-body">
                <div id="carouselExampleControls" class="carousel slide" data-bs-ride="carousel">
                    <div class="carousel-inner">';

    $first = true;
    foreach ($roomPictures as $picture) {
        echo '<div class="carousel-item ' . ($first ? 'active' : '') . '">
                <img src="' . $picture . '" class="d-block w-100" style="width: 1920px; height: 768px" alt="Room Picture">
              </div>';
        $first = false;
    }

    echo '
                    </div>
                    <a class="carousel-control-prev" href="#carouselExampleControls" role="button" data-bs-slide="prev">
                        <span class="carousel-control-prev-icon" aria-hidden="true"></span>
                        <span class="visually-hidden">Previous</span>
                    </a>
                    <a class="carousel-control-next" href="#carouselExampleControls" role="button" data-bs-slide="next">
                        <span class="carousel-control-next-icon" aria-hidden="true"></span>
                        <span class="visually-hidden">Next</span>
                    </a>
                </div>
            </div>
        </div>
    </div>';
} else {
    echo "<p>No pictures available for this room.</p>";
}


        // Display room amenities
        echo "<h2>Room Amenities</h2>";
        if (!empty($roomAmenities)) {
            echo "<table border='1' cellpadding='10' cellspacing='0' class='table table-striped table-bordered'>";
            echo "<tr><th>Amenity</th></tr>";
            foreach ($roomAmenities as $amenity) {
                echo "<tr><td>{$amenity}</td></tr>";
            }
            echo "</table>";
        } else {
            echo "<p>No amenities listed for this room.</p>";
        }

        // Display room videos
        echo "<h2>Room Videos</h2>";
        if (!empty($roomVideos)) {
            echo "<table border='1' cellpadding='10' cellspacing='0' class='table table-striped table-bordered'>";
            echo "<tr><th>Video</th></tr>";
            foreach ($roomVideos as $video) {
                echo "<tr><td><video width='320' height='240' controls>
                        <source src='{$video}' type='video/mp4'>
                      Your browser does not support the video tag.
                      </video></td></tr>";
            }
            echo "</table>";
        } else {
            echo "<p>No videos available for this room.</p>";
        }

    } else {
        echo '<p>No records found for the specified room.</p>';
    }
}

// Close the database connection
$conn->close();
?>
